import React from 'react';
import { useTheme } from '../../contexts/ThemeContext';
import { FaFilter, FaTimes } from 'react-icons/fa';

const EmployeeFilters = ({ 
  divisions = [],
  roles = [],
  projects = [],
  selectedDivision,
  selectedRole,
  selectedProject,
  onDivisionChange,
  onRoleChange,
  onProjectChange,
  filtersLoading = false
}) => {
  const { isLight } = useTheme();

  // Theme-based styling
  const cardBgClass = isLight 
    ? 'bg-white border border-yellow-200 shadow-md hover:shadow-lg' 
    : 'bg-gradient-to-br from-gray-800 to-slate-800 border border-gray-700 shadow-lg hover:shadow-xl';
  
  const textClass = isLight 
    ? 'text-amber-700' 
    : 'text-gray-300';

  // Check if any filters are active
  const hasActiveFilters = selectedDivision !== 'All' || selectedRole !== 'All' || selectedProject !== 'All';

  // Clear all filters
  const clearAllFilters = () => {
    onDivisionChange('All');
    onRoleChange('All');
    onProjectChange('All');
  };

  return (
    <div className={`${cardBgClass} rounded-xl px-4 sm:px-6 py-3 sm:py-4 transition-all duration-300`}>
      <div className="flex flex-col lg:flex-row lg:items-center gap-4 lg:gap-6">
        {/* Filter Label */}
        <div className="flex items-center gap-2 flex-shrink-0">
          <FaFilter className={`text-sm ${isLight ? 'text-amber-600' : 'text-gray-400'}`} />
          <span className={`text-sm font-medium ${textClass}`}>Filter Employees:</span>
        </div>
        
        {/* Filter Dropdowns */}
        <div className="flex flex-col sm:flex-row gap-3 lg:gap-4 flex-1 min-w-0">
          {/* Division Filter */}
          <div className="flex-1 min-w-0">
            <label className={`block text-xs font-medium ${textClass} mb-1`}>Division</label>
            <select
              value={selectedDivision}
              onChange={(e) => onDivisionChange(e.target.value)}
              disabled={filtersLoading}
              className={`w-full px-3 py-2 rounded-lg border text-sm transition-all duration-200 ${
                isLight 
                  ? 'bg-white text-gray-900 border-gray-300 focus:ring-yellow-500 focus:border-yellow-500' 
                  : 'bg-gray-700 text-gray-100 border-gray-600 focus:ring-gray-400 focus:border-gray-400'
              } ${filtersLoading ? 'opacity-50 cursor-not-allowed' : ''}`}
            >
              <option value="All">All Divisions</option>
              {divisions.map(division => (
                <option key={division.division_id} value={division.division_name}>
                  {division.division_name}
                </option>
              ))}
            </select>
          </div>

          {/* Role Filter */}
          <div className="flex-1 min-w-0">
            <label className={`block text-xs font-medium ${textClass} mb-1`}>Role</label>
            <select
              value={selectedRole}
              onChange={(e) => onRoleChange(e.target.value)}
              disabled={filtersLoading}
              className={`w-full px-3 py-2 rounded-lg border text-sm transition-all duration-200 ${
                isLight 
                  ? 'bg-white text-gray-900 border-gray-300 focus:ring-yellow-500 focus:border-yellow-500' 
                  : 'bg-gray-700 text-gray-100 border-gray-600 focus:ring-gray-400 focus:border-gray-400'
              } ${filtersLoading ? 'opacity-50 cursor-not-allowed' : ''}`}
            >
              <option value="All">All Roles</option>
              {roles.map(role => (
                <option key={role.role_id} value={role.role_name}>
                  {role.role_name}
                </option>
              ))}
            </select>
          </div>

          {/* Project Filter */}
          <div className="flex-1 min-w-0">
            <label className={`block text-xs font-medium ${textClass} mb-1`}>Project</label>
            <select
              value={selectedProject}
              onChange={(e) => onProjectChange(e.target.value)}
              disabled={filtersLoading}
              className={`w-full px-3 py-2 rounded-lg border text-sm transition-all duration-200 ${
                isLight 
                  ? 'bg-white text-gray-900 border-gray-300 focus:ring-yellow-500 focus:border-yellow-500' 
                  : 'bg-gray-700 text-gray-100 border-gray-600 focus:ring-gray-400 focus:border-gray-400'
              } ${filtersLoading ? 'opacity-50 cursor-not-allowed' : ''}`}
            >
              <option value="All">All Projects</option>
              <option value="No Projects">No Projects</option>
              {projects.map(project => (
                <option key={project.project_id} value={project.project_name}>
                  {project.project_name}
                </option>
              ))}
            </select>
          </div>

          {/* Clear Filters Button */}
          {hasActiveFilters && (
            <div className="flex-shrink-0 self-end">
              <button
                onClick={clearAllFilters}
                className={`flex items-center gap-2 px-3 py-2 rounded-lg text-sm font-medium transition-all duration-200 hover:scale-105 ${
                  isLight
                    ? 'bg-red-100 hover:bg-red-200 text-red-700 border border-red-200'
                    : 'bg-red-900/20 hover:bg-red-900/30 text-red-400 border border-red-800'
                }`}
                title="Clear all filters"
              >
                <FaTimes className="text-xs" />
                <span>Clear</span>
              </button>
            </div>
          )}
        </div>
      </div>

      
    </div>
  );
};

export default EmployeeFilters; 