import React, { useState, useEffect, useMemo } from "react";
import {
  FaHome,
  FaUsers,
  FaProjectDiagram,
  FaUserTie,
  FaBuilding,
  FaClipboardList,
  FaTasks,
  FaFileAlt,
} from "react-icons/fa";
import { useTheme } from "../contexts/ThemeContext";

const menuItems = [
  { id: "dashboard", label: "Dashboard", icon: <FaHome /> },
  { id: "employees", label: "Employees", icon: <FaUsers /> },
  { id: "employee_log", label: "Employee Log", icon: <FaClipboardList /> },
  { id: "projects", label: "Projects", icon: <FaProjectDiagram /> },
  { id: "project_log", label: "Project Log", icon: <FaTasks /> },
  { id: "forms", label: "Forms", icon: <FaFileAlt /> },
  { id: "project_lead", label: "Project Lead", icon: <FaUserTie /> },
  { id: "departments", label: "Departments", icon: <FaBuilding /> },
];

const Sidebar = ({
  activeSection,
  onSectionChange,
  onSidebarToggle,
  isMobileMenuOpen,
  onMobileMenuClose,
}) => {
  const [isExpanded, setIsExpanded] = useState(false);
  const { isLight } = useTheme();

  // Notify parent component when sidebar state changes
  useEffect(() => {
    if (onSidebarToggle) {
      onSidebarToggle(isExpanded);
    }
  }, [isExpanded, onSidebarToggle]);

  const handleItemClick = (itemId) => {
    onSectionChange(itemId);

    // Close mobile menu when item is clicked
    if (onMobileMenuClose) {
      onMobileMenuClose();
    }
  };

  const handleMouseEnter = () => {
    // Only expand on hover for desktop
    if (window.innerWidth >= 1024) {
      setIsExpanded(true);
    }
  };

  const handleMouseLeave = () => {
    // Only collapse on leave for desktop
    if (window.innerWidth >= 1024) {
      setIsExpanded(false);
    }
  };

  const handleOverlayClick = () => {
    if (onMobileMenuClose) {
      onMobileMenuClose();
    }
  };

  // Theme-based styling
  const sidebarBg = isLight ? "bg-yellow-400" : "bg-slate-800";

  const textColor = isLight ? "text-gray-800" : "text-slate-100";

  const getItemStyles = useMemo(() => {
    return (isActive) => {
      if (isActive) {
        return isLight
          ? "bg-white/30 text-gray-900 shadow-sm"
          : "bg-blue-600 text-white shadow-lg border-l-4 border-cyan-400";
      }
      return isLight
        ? "text-gray-800 hover:bg-white/20 active:bg-white/30"
        : "text-slate-200 hover:bg-slate-700 hover:text-cyan-300 active:bg-blue-900 transition-all duration-300";
    };
  }, [isLight]);

  return (
    <>
      {/* Mobile Overlay - Only visible when mobile menu is open */}
      {isMobileMenuOpen && (
        <div
          className="fixed inset-0 bg-transparent backdrop-blur-[1px] z-30 lg:hidden"
          onClick={handleOverlayClick}
        />
      )}

      {/* Sidebar */}
      <div
        className={`group fixed top-16 z-50 transition-transform duration-300 ease-in-out h-full ${
          // Mobile: positioned on right, slides from right
          // Desktop: positioned on left, always visible
          `right-0 lg:left-0 lg:right-auto ${
            isMobileMenuOpen ? "translate-x-0" : "translate-x-full"
          } lg:translate-x-0`
        }`}
        onMouseEnter={handleMouseEnter}
        onMouseLeave={handleMouseLeave}
      >
        <div
          className={`${sidebarBg} transition-all duration-300 ease-in-out h-full flex flex-col overflow-hidden shadow-xl ${
            // Mobile: full width when open, desktop: responsive width with hover
            isMobileMenuOpen
              ? "w-64 lg:w-12 lg:group-hover:w-48"
              : "w-12 group-hover:w-48"
          }`}
        >
          {/* Decorative gradient overlay for light mode */}
          {isLight && (
            <div className="absolute inset-0 bg-gradient-to-br from-white/10 via-transparent to-amber-600/20 pointer-events-none"></div>
          )}

          {menuItems.map((item, index) => (
            <div
              key={index}
              className={`relative flex items-center gap-3 p-4 cursor-pointer transition-all duration-200 touch-manipulation ${getItemStyles(
                activeSection === item.id
              )}`}
              onClick={() => handleItemClick(item.id)}
            >
              <span className="text-xl flex-shrink-0 relative z-10 flex items-center justify-center w-6 h-6">
                {item.icon}
              </span>
              <span
                className={`whitespace-nowrap font-medium transition-opacity duration-300 relative z-10 ${
                  isMobileMenuOpen
                    ? "opacity-100 lg:opacity-0 lg:group-hover:opacity-100"
                    : "opacity-0 group-hover:opacity-100"
                }`}
              >
                {item.label}
              </span>
            </div>
          ))}

          {/* Bottom decorative element for light mode */}
          {isLight && (
            <div className="mt-auto p-4">
              <div className="h-px bg-gradient-to-r from-transparent via-white/30 to-transparent"></div>
            </div>
          )}
        </div>
      </div>
    </>
  );
};

export default Sidebar;
